/*****************************************************************************/
/*  Includes                                                                 */
/*****************************************************************************/
#include "netx51_canctrl.h"
#include "hal_resources_defines_netx51.h"

/*****************************************************************************/
/*  Variables                                                                */
/*****************************************************************************/
__USE_CANCTRL

/*****************************************************************************/
/*  Functions                                                                */
/*****************************************************************************/

/*****************************************************************************/
/*! Enter Reset Mode
* \description
*   Enters the reset mode of the CAN controller and sets it into PeliCAN mode.
* \class 
*   Control Service Class
* \params
* \return
*                                                                            */
/*****************************************************************************/
void NX51_CanCtrl_EnterResetMode( void )
{
  s_ptCanCtrl->ulCanctrl_mode |= MSK_NX51_canctrl_mode_reset_mode;
  s_ptCanCtrl->ulCanctrl_mode_control = MSK_NX51_canctrl_mode_control_mode;
}

/*****************************************************************************/
/*! Leave Reset Mode
* \description
*   Leavs the reset mode of the CAN controller and goes to PeliCAN operating mode.
* \class 
*   Control Service Class
* \params
* \return
*                                                                            */
/*****************************************************************************/
void NX51_CanCtrl_LeaveResetMode( void )
{
  s_ptCanCtrl->ulCanctrl_mode_control = MSK_NX51_canctrl_mode_control_mode;
  s_ptCanCtrl->ulCanctrl_mode &= ~MSK_NX51_canctrl_mode_reset_mode;
}

/*****************************************************************************/
/*! Set Baudrate
* \description
*   Configures the CAN controller with a specific speed.
* \class 
*   Control Service Class
* \params
*   ulBaudrate        [in]    Baud rate in kbit/s
*   uOversampling     [in]    0/1: Oversampling (three-times) is disabled/enabled
* \return
*   NX51_CANCTRL_RESULT_OKAY  
*   NX51_CANCTRL_RESULT_WRONG_MODE
*   NX51_CANCTRL_RESULT_INVAL_PARAM                                          */
/*****************************************************************************/
NX51_CANCTRL_RESULT_E NX51_CanCtrl_SetBaudrate( uint32_t     ulBaudrate,
                                                unsigned int uOversampling )
{
  uint32_t  ulBtr0;
  uint32_t  ulBtr1;

  if ((s_ptCanCtrl->ulCanctrl_mode & MSK_NX51_canctrl_mode_reset_mode) == 0)
    return NX51_CANCTRL_RESULT_WRONG_MODE;

  switch( ulBaudrate )
  {
  case NX51_CANCTRL_BAUDRATE_1000K:
    ulBtr0 = ((uint32_t)10 << SRT_NX51_canctrl_bus_timing0_prescaler);
    ulBtr1 = ((uint32_t)7 << SRT_NX51_canctrl_bus_timing1_tseg1) |
             ((uint32_t)uOversampling << SRT_NX51_canctrl_bus_timing1_oversampling);
    break;
  case NX51_CANCTRL_BAUDRATE_800K :
    ulBtr0 = ((uint32_t)25 << SRT_NX51_canctrl_bus_timing0_prescaler);
    ulBtr1 = ((uint32_t)2 << SRT_NX51_canctrl_bus_timing1_tseg1) |
             ((uint32_t)uOversampling << SRT_NX51_canctrl_bus_timing1_oversampling);
    break;
  case NX51_CANCTRL_BAUDRATE_500K :
    ulBtr0 = ((uint32_t)20 << SRT_NX51_canctrl_bus_timing0_prescaler);
    ulBtr1 = ((uint32_t)7 << SRT_NX51_canctrl_bus_timing1_tseg1) |
             ((uint32_t)uOversampling << SRT_NX51_canctrl_bus_timing1_oversampling);
     break;
  case NX51_CANCTRL_BAUDRATE_250K :
    ulBtr0 = ((uint32_t)40 << SRT_NX51_canctrl_bus_timing0_prescaler);
    ulBtr1 = ((uint32_t)7 << SRT_NX51_canctrl_bus_timing1_tseg1) |
             ((uint32_t)uOversampling << SRT_NX51_canctrl_bus_timing1_oversampling);
    break;
  case NX51_CANCTRL_BAUDRATE_125K :
    ulBtr0 = ((uint32_t)80 << SRT_NX51_canctrl_bus_timing0_prescaler);
    ulBtr1 = ((uint32_t)7 << SRT_NX51_canctrl_bus_timing1_tseg1) |
             ((uint32_t)uOversampling << SRT_NX51_canctrl_bus_timing1_oversampling);
    break;
  case NX51_CANCTRL_BAUDRATE_50K  :
    ulBtr0 = ((uint32_t)200 << SRT_NX51_canctrl_bus_timing0_prescaler);
    ulBtr1 = ((uint32_t)7 << SRT_NX51_canctrl_bus_timing1_tseg1) |
             ((uint32_t)uOversampling << SRT_NX51_canctrl_bus_timing1_oversampling);
    break;
  case NX51_CANCTRL_BAUDRATE_20K  :
    ulBtr0 = ((uint32_t)500 << SRT_NX51_canctrl_bus_timing0_prescaler);
    ulBtr1 = ((uint32_t)7 << SRT_NX51_canctrl_bus_timing1_tseg1) |
             ((uint32_t)uOversampling << SRT_NX51_canctrl_bus_timing1_oversampling);
    break;
  case NX51_CANCTRL_BAUDRATE_10K  :
    ulBtr0 = ((uint32_t)500 << SRT_NX51_canctrl_bus_timing0_prescaler);
    ulBtr1 = ((uint32_t)16 << SRT_NX51_canctrl_bus_timing1_tseg1) | (1 << SRT_NX51_canctrl_bus_timing1_tseg2) |
             ((uint32_t)uOversampling << SRT_NX51_canctrl_bus_timing1_oversampling);
    break;
  default: return NX51_CANCTRL_RESULT_INVAL_PARAM;
  }

  s_ptCanCtrl->aulCanctrl_bus_timing[0] = ulBtr0;
  s_ptCanCtrl->aulCanctrl_bus_timing[1] = ulBtr1;

  return NX51_CANCTRL_RESULT_OKAY;
}

/*****************************************************************************/
/*! Set Single Acceptance Filter
* \description
*   Sets a single 32-bit acceptance filter
* \class 
*   Control Service Class
* \params
*   ptFilter        [in]    Pointer to acceptance filter configuration (NX51_CANCTRL_SINGLEFILTER_T)
* \return
*   NX51_CANCTRL_RESULT_OKAY        
*   NX51_CANCTRL_RESULT_WRONG_MODE                                           */
/*****************************************************************************/
NX51_CANCTRL_RESULT_E NX51_CanCtrl_SetSingleAcceptanceFilter( NX51_CANCTRL_SINGLEFILTER_T* ptFilter )
{
  if ((s_ptCanCtrl->ulCanctrl_mode & MSK_NX51_canctrl_mode_reset_mode) == 0)
    return NX51_CANCTRL_RESULT_WRONG_MODE;

  s_ptCanCtrl->aulCanctrl_data[3] = ptFilter->ulAcceptanceCode & 0xff;
  s_ptCanCtrl->aulCanctrl_data[2] = (ptFilter->ulAcceptanceCode >> 8) & 0xff;
  s_ptCanCtrl->aulCanctrl_data[1] = (ptFilter->ulAcceptanceCode >> 16) & 0xff;
  s_ptCanCtrl->aulCanctrl_data[0] = (ptFilter->ulAcceptanceCode >> 24) & 0xff;
  /* Note: In netX51/52 there is a failure when writing to Acceptance Mask 0 and 1.
     The write content of Acceptance Mask 0 goes correctly into register Acceptance Mask 0 and additionally abnormal into register Acceptance Mask 1.
     Acceptance Mask 1 is not writable individually.
     Because of this bug the content of Acceptance Mask 0 and Acceptance Mask 1 must be the same and that leads to restricted filter possibilities.
     Furthermore the locations of Acceptance Mask 2/3 changed to Registers Data 5 and Data 6.
  */
  s_ptCanCtrl->aulCanctrl_data[6] = ptFilter->ulAcceptanceMask & 0xff;
  s_ptCanCtrl->aulCanctrl_data[5] = (ptFilter->ulAcceptanceMask >> 8) & 0xff;
  /*s_ptCanCtrl->aulCanctrl_data[4] = (ptFilter->ulAcceptanceMask >> 16) & 0xff; */ /* disabled: bug!!! */
  s_ptCanCtrl->aulCanctrl_data[4] = (ptFilter->ulAcceptanceMask >> 24) & 0xff;

  s_ptCanCtrl->ulCanctrl_mode |= MSK_NX51_canctrl_mode_acceptance_mode;

  return NX51_CANCTRL_RESULT_OKAY;
}

/*****************************************************************************/
/*! Get Single Acceptance Filter
* \description
*   Gets the configured single 32-bit acceptance filter
* \class
*   Status Service Class
* \params
*   ptFilter        [out]    Pointer to acceptance filter configuration (NX51_CANCTRL_SINGLEFILTER_T)
* \return
*   NX51_CANCTRL_RESULT_OKAY
*   NX51_CANCTRL_RESULT_WRONG_MODE                                           */
/*****************************************************************************/
NX51_CANCTRL_RESULT_E NX51_CanCtrl_GetSingleAcceptanceFilter( NX51_CANCTRL_SINGLEFILTER_T* ptFilter )
{
  if ((s_ptCanCtrl->ulCanctrl_mode & MSK_NX51_canctrl_mode_reset_mode) == 0)
    return NX51_CANCTRL_RESULT_WRONG_MODE;

  if ((s_ptCanCtrl->ulCanctrl_mode & MSK_NX51_canctrl_mode_acceptance_mode) == 0)
    return NX51_CANCTRL_RESULT_WRONG_MODE;

  ptFilter->ulAcceptanceCode = s_ptCanCtrl->aulCanctrl_data[3] |
                               (s_ptCanCtrl->aulCanctrl_data[2] << 8) |
                               (s_ptCanCtrl->aulCanctrl_data[1] << 16) |
                               (s_ptCanCtrl->aulCanctrl_data[0] << 24);

  ptFilter->ulAcceptanceMask = s_ptCanCtrl->aulCanctrl_data[7] |
                               (s_ptCanCtrl->aulCanctrl_data[6] << 8) |
                               (s_ptCanCtrl->aulCanctrl_data[5] << 16) |
                               (s_ptCanCtrl->aulCanctrl_data[4] << 24);
  return NX51_CANCTRL_RESULT_OKAY;
}

/*****************************************************************************/
/*! Set Dual Acceptance Filter
* \description
*   Sets two 16-bit acceptance filters
* \class 
*   Control Service Class
* \params
*   ptFilter        [in]    Pointer to acceptance filter configuration (NX51_CANCTRL_DUALFILTER_T)
* \return
*   NX51_CANCTRL_RESULT_OKAY        
*   NX51_CANCTRL_RESULT_WRONG_MODE                                           */
/*****************************************************************************/
NX51_CANCTRL_RESULT_E NX51_CanCtrl_SetDualAcceptanceFilter( NX51_CANCTRL_DUALFILTER_T* ptFilter )
{
  if ((s_ptCanCtrl->ulCanctrl_mode & MSK_NX51_canctrl_mode_reset_mode) == 0)
    return NX51_CANCTRL_RESULT_WRONG_MODE;

  s_ptCanCtrl->aulCanctrl_data[1] = (uint32_t)ptFilter->usAcceptanceCode0 & 0xff;
  s_ptCanCtrl->aulCanctrl_data[0] = ((uint32_t)ptFilter->usAcceptanceCode0 >> 8) & 0xff;
  s_ptCanCtrl->aulCanctrl_data[3] = (uint32_t)ptFilter->usAcceptanceCode1 & 0xff;
  s_ptCanCtrl->aulCanctrl_data[2] = ((uint32_t)ptFilter->usAcceptanceCode1 >> 8) & 0xff;

  /* Note: In netX51/52 there is a failure when writing to Acceptance Mask 0 and 1.
     The write content of Acceptance Mask 0 goes correctly into register Acceptance Mask 0 and additionally abnormal into register Acceptance Mask 1.
     Acceptance Mask 1 is not writable individually.
     Because of this bug the content of Acceptance Mask 0 and Acceptance Mask 1 must be the same and that leads to restricted filter possibilities.
     Furthermore the locations of Acceptance Mask 2/3 changed to Registers Data 5 and Data 6.
  */
  s_ptCanCtrl->aulCanctrl_data[4] = (uint32_t)ptFilter->usAcceptanceMask0 & 0xff;
  /*s_ptCanCtrl->aulCanctrl_data[4] = ((uint32_t)ptFilter->usAcceptanceMask0 >> 8) & 0xff; */ /* disabled: bug!!! */
  s_ptCanCtrl->aulCanctrl_data[6] = (uint32_t)ptFilter->usAcceptanceMask1 & 0xff;
  s_ptCanCtrl->aulCanctrl_data[5] = ((uint32_t)ptFilter->usAcceptanceMask1 >> 8) & 0xff;

  s_ptCanCtrl->ulCanctrl_mode &= ~MSK_NX51_canctrl_mode_acceptance_mode;

  return NX51_CANCTRL_RESULT_OKAY;
}

/*****************************************************************************/
/*! Get Dual Acceptance Filter
* \description
*   Gets the configured two 16-bit acceptance filters
* \class
*   Status Service Class
* \params
*   ptFilter        [out]    Pointer to acceptance filter configuration (NX51_CANCTRL_DUALFILTER_T)
* \return
*   NX51_CANCTRL_RESULT_OKAY
*   NX51_CANCTRL_RESULT_WRONG_MODE                                           */
/*****************************************************************************/
NX51_CANCTRL_RESULT_E NX51_CanCtrl_GetDualAcceptanceFilter( NX51_CANCTRL_DUALFILTER_T* ptFilter )
{
  if ((s_ptCanCtrl->ulCanctrl_mode & MSK_NX51_canctrl_mode_reset_mode) == 0)
    return NX51_CANCTRL_RESULT_WRONG_MODE;

  if ((s_ptCanCtrl->ulCanctrl_mode & MSK_NX51_canctrl_mode_acceptance_mode) != 0)
    return NX51_CANCTRL_RESULT_WRONG_MODE;

  ptFilter->usAcceptanceCode0 = s_ptCanCtrl->aulCanctrl_data[1] |
                               (s_ptCanCtrl->aulCanctrl_data[0] << 8);

  ptFilter->usAcceptanceCode1 = s_ptCanCtrl->aulCanctrl_data[3] |
                               (s_ptCanCtrl->aulCanctrl_data[2] << 8);

  ptFilter->usAcceptanceMask0 = s_ptCanCtrl->aulCanctrl_data[5] |
                               (s_ptCanCtrl->aulCanctrl_data[4] << 8);

  ptFilter->usAcceptanceMask1 = s_ptCanCtrl->aulCanctrl_data[7] |
                               (s_ptCanCtrl->aulCanctrl_data[6] << 8);

  return NX51_CANCTRL_RESULT_OKAY;
}

/*****************************************************************************/
/*! Enter Listen Mode
* \description
*   Enters the listen mode of the CAN controller.
* \class 
*   Control Service Class
* \params
* \return                                                                    */
/*****************************************************************************/
void NX51_CanCtrl_EnterListenMode( void )
{
  s_ptCanCtrl->ulCanctrl_mode |= MSK_NX51_canctrl_mode_listen_mode;
}


/*****************************************************************************/
/*! Leave Listen Mode
* \description
*   Leavs the listen mode of the CAN controller and goes to operating mode.
* \class 
*   Control Service Class
* \params
* \return                                                                    */
/*****************************************************************************/
void NX51_CanCtrl_LeaveListenMode( void )
{
  s_ptCanCtrl->ulCanctrl_mode &= ~MSK_NX51_canctrl_mode_listen_mode;
}


/*****************************************************************************/
/*! Enter Selftest Mode
* \description
*   Enters the selftest mode of the CAN controller
* \class 
*   Control Service Class
* \params
* \return                                                                    */
/*****************************************************************************/
void NX51_CanCtrl_EnterSelftestMode( void )
{
  s_ptCanCtrl->ulCanctrl_mode |= MSK_NX51_canctrl_mode_selftest;
}


/*****************************************************************************/
/*! Leave Selftest Mode
* \description
*   Leaves the selftest mode of the CAN controller and goes to operating mode.
* \class 
*   Control Service Class
* \params
* \return                                                                    */
/*****************************************************************************/
void NX51_CanCtrl_LeaveSelftestMode( void )
{
  s_ptCanCtrl->ulCanctrl_mode &= ~MSK_NX51_canctrl_mode_selftest;
}

/*****************************************************************************/
/*! Send Message
* \description
*   Sends a message on CAN bus.
* \class 
*   Transmission Service Class
* \params
*   ptMessage        [in]   Pointer to message (NX51_CANCTRL_MESSAGE_T)
*   fSingleShot      [in]   Send message in single-shot (0) or repeat if no acknowledge was received (1)
*   fSelfRx          [in]   Send message and receive it simultaneously
* \return
*   NX51_CANCTRL_RESULT_OKAY                  
*   NX51_CANCTRL_RESULT_TX_BUFFER_NOT_FREE   
*   NX51_CANCTRL_RESULT_WRONG_MODE                                             */
/*****************************************************************************/
NX51_CANCTRL_RESULT_E NX51_CanCtrl_SendMessage( NX51_CANCTRL_MESSAGE_T* ptMessage,
                                                bool                    fSingleShot,
                                                bool                    fSelfRx )
{
  uint32_t  ulCmd;
  uint32_t  ulDataPtr;
  uint32_t  ulLength;
  uint32_t  ulIdx;

  if ((s_ptCanCtrl->ulCanctrl_mode & MSK_NX51_canctrl_mode_reset_mode) != 0)
    return NX51_CANCTRL_RESULT_WRONG_MODE;

  if ((s_ptCanCtrl->ulCanctrl_status & MSK_NX51_canctrl_status_tx_buf_status) == 0)
    return NX51_CANCTRL_RESULT_TX_BUFFER_NOT_FREE;
  
  /* build message */
  s_ptCanCtrl->aulCanctrl_data[0] = ((uint32_t)ptMessage->uFrameFormat << 7) |
                                    ((uint32_t)ptMessage->uRtr << 6) |
                                    (uint32_t)ptMessage->uDlc;

  if (ptMessage->uFrameFormat == 0)
  {
    s_ptCanCtrl->aulCanctrl_data[1] = (ptMessage->ulIdentifier >> 3) & 0xff;
    s_ptCanCtrl->aulCanctrl_data[2] = (ptMessage->ulIdentifier << 5) & 0xff;
    ulDataPtr = 3;
  }
  else
  {
    s_ptCanCtrl->aulCanctrl_data[1] = (ptMessage->ulIdentifier >> 21) & 0xff;
    s_ptCanCtrl->aulCanctrl_data[2] = (ptMessage->ulIdentifier >> 13) & 0xff;
    s_ptCanCtrl->aulCanctrl_data[3] = (ptMessage->ulIdentifier >> 5)  & 0xff;
    s_ptCanCtrl->aulCanctrl_data[4] = (ptMessage->ulIdentifier << 3)  & 0xff;
    ulDataPtr = 5;
  }
  
  if( ptMessage->uRtr == 0 )
  {
    ulLength = ptMessage->uDlc;
    if( ulLength > 8 ) ulLength = 8;
    
    for( ulIdx = 0; ulIdx < ulLength; ulIdx++ )
      s_ptCanCtrl->aulCanctrl_data[ulDataPtr+ulIdx] = ptMessage->abData[ulIdx];
  }

  /* send message */
  if( !fSelfRx )
    ulCmd = MSK_NX51_canctrl_command_tx_request;
  else
    ulCmd = MSK_NX51_canctrl_command_self_rx_request;

  if( fSingleShot )
    ulCmd |= MSK_NX51_canctrl_command_abort_tx;

  s_ptCanCtrl->ulCanctrl_command = ulCmd;

  return NX51_CANCTRL_RESULT_OKAY;
}

/*****************************************************************************/
/*! Abort Message
* \description
*   Aborts a currently pending message.
* \class 
*   Transmission Service Class
* \params
* \return                                                                    */
/*****************************************************************************/
void NX51_CanCtrl_AbortMessage( void )
{
  s_ptCanCtrl->ulCanctrl_command |= MSK_NX51_canctrl_command_abort_tx;
}

/*****************************************************************************/
/*! Receive Message
* \description
*   Receives the first pending message from the RX FIFO.
* \class 
*   Reception Service Class
* \params
*   ptMessage     [out]   Pointer to received message
* \return
*   NX51_CANCTRL_RESULT_OKAY 
*   NX51_CANCTRL_RESULT_NO_MESSAGE_AVAILABLE
*   NX51_CANCTRL_RESULT_MESSAGE_DROPPED
*   NX51_CANCTRL_RESULT_WRONG_MODE                                             */
/*****************************************************************************/
NX51_CANCTRL_RESULT_E NX51_CanCtrl_ReceiveMessage( NX51_CANCTRL_MESSAGE_T* ptMessage )
{
  uint32_t  ulDataPtr;
  uint32_t  ulLength;
  uint32_t  ulIdx;

  /* check that we are operational */
  if( (s_ptCanCtrl->ulCanctrl_mode & MSK_NX51_canctrl_mode_reset_mode) != 0 )
    return NX51_CANCTRL_RESULT_WRONG_MODE;
  
  /* check that message available */
  if( (s_ptCanCtrl->ulCanctrl_status & MSK_NX51_canctrl_status_rx_buf_status) == 0 )
    return NX51_CANCTRL_RESULT_NO_MESSAGE_AVAILABLE;

  /* check for overrun */
  if( (s_ptCanCtrl->ulCanctrl_status & MSK_NX51_canctrl_status_overrun) != 0 ) {
    /* clear overrun condition */
    s_ptCanCtrl->ulCanctrl_command = MSK_NX51_canctrl_command_clr_overrun | MSK_NX51_canctrl_command_release_rx_buf;
    return NX51_CANCTRL_RESULT_MESSAGE_DROPPED;
  }

  /* get message */
  ptMessage->uFrameFormat = (s_ptCanCtrl->aulCanctrl_data[0] >> 7) & 0x1;

  if (ptMessage->uFrameFormat == 0)
  {
    ptMessage->uRtr = (s_ptCanCtrl->aulCanctrl_data[0] >> 6) & 0x1;
    ptMessage->ulIdentifier = ( (s_ptCanCtrl->aulCanctrl_data[1] << 24)
                             | ((s_ptCanCtrl->aulCanctrl_data[2] & 0xef) << 16)
                              )>>21;
    ulDataPtr = 3;
  }
  else
  {
    ptMessage->uRtr = (s_ptCanCtrl->aulCanctrl_data[0] >> 6) & 0x1;
    ptMessage->ulIdentifier = ( (s_ptCanCtrl->aulCanctrl_data[1] << 24)
                               |(s_ptCanCtrl->aulCanctrl_data[2] << 16)
                               |(s_ptCanCtrl->aulCanctrl_data[3] << 8)
                               |(s_ptCanCtrl->aulCanctrl_data[4] << 0)
                              )>>3;
    ulDataPtr = 5;
  }
  
  ptMessage->uDlc = s_ptCanCtrl->aulCanctrl_data[0] & 0xf;
  
  if( ptMessage->uRtr == 0 )
  {
    ulLength = ptMessage->uDlc;
    if( ulLength > 8 ) ulLength = 8;
    
    for( ulIdx = 0; ulIdx < ulLength; ulIdx++ )
      ptMessage->abData[ulIdx] = s_ptCanCtrl->aulCanctrl_data[ulDataPtr+ulIdx];
  }
  
  /* release buffer */
  s_ptCanCtrl->ulCanctrl_command = MSK_NX51_canctrl_command_release_rx_buf;

  return NX51_CANCTRL_RESULT_OKAY;
}

/*****************************************************************************/
/*! Get RX Message Count
* \description
*   Returns the number of pending messages in the RX FIFO.
* \class 
*   Status Service Class
* \params
* \return
*   Current number of messages in rx fifo                                    */
/*****************************************************************************/
uint32_t NX51_CanCtrl_GetRxMessageCnt( void )
{
  return s_ptCanCtrl->ulCanctrl_rx_message_cnt;
}

/*****************************************************************************/
/*! Get Status
* \description
*   Returns the status of the CAN controller.
* \class 
*   Status Service Class
* \params
* \return
*   Current status of CAN controller                                         */
/*****************************************************************************/
uint32_t NX51_CanCtrl_GetStatus( void )
{
  return s_ptCanCtrl->ulCanctrl_status;
}

/*****************************************************************************/
/*! Get and Clear IRQ Status
* \description
*   Returns and clears the interrupt status of the CAN controller.
*   The status of the receive interrupt bit will not be cleared.
* \class 
*   Status Service Class
* \params
* \return
*   Current interrupt status value                                           */
/*****************************************************************************/
uint32_t NX51_CanCtrl_GetAndClearIrqStatus( void )
{
  return s_ptCanCtrl->ulCanctrl_irq;
}

/*****************************************************************************/
/*! Set IRQ Mask
* \description
*   Sets the IRQ enable mask.
* \class 
*   Control Service Class
* \params
*   ulIrqMask        [in]   Interrupt enable mask
* \return                                                                    */
/*****************************************************************************/
void NX51_CanCtrl_SetIrqMask( uint32_t ulIrqMask )
{
  s_ptCanCtrl->ulCanctrl_irq_en = ulIrqMask;
}

/*****************************************************************************/
/*! Get IRQ Mask
* \description
*   Returns the currently set IRQ enable mask.
* \class 
*   Status Service Class
* \params
* \return
*   Current irq_mask value                                                   */
/*****************************************************************************/
uint32_t NX51_CanCtrl_GetIrqMask( void )
{
  return s_ptCanCtrl->ulCanctrl_irq_en;
}

/*****************************************************************************/
/*! Get and Clear Arbitration Lost Status
* \description
*   Returns and clears the status of the last latched lost
*   bus arbitration.
* \class 
*   Status Service Class
* \params
* \return
*   Current status of the last latched arbitration lost                      */
/*****************************************************************************/
uint32_t NX51_CanCtrl_GetAndClearArblostStatus( void )
{
  return s_ptCanCtrl->ulCanctrl_arb_lost_capture;
}

/*****************************************************************************/
/*! Get and Clear Error Code
* \description
*   Returns and clears the error code of the last latched bus error.
* \class 
*   Status Service Class
* \params
* \return
*   Current error code of the last latched bus error                         */
/*****************************************************************************/
uint32_t NX51_CanCtrl_GetAndClearErrorCode( void )
{
  return s_ptCanCtrl->ulCanctrl_err_code_capture;
}

/*****************************************************************************/
/*! Set Error Warning Limit
* \description
*   Sets the warning limit for error counters.
* \class 
*   Control Service Class
* \params
*   ulLimit        [in]   Error warning limit
* \return
*   NX51_CANCTRL_RESULT_OKAY      
*   NX51_CANCTRL_RESULT_WRONG_MODE                                           */
/*****************************************************************************/
NX51_CANCTRL_RESULT_E NX51_CanCtrl_SetWarningLimit( uint32_t ulLimit )
{
  if ((s_ptCanCtrl->ulCanctrl_mode & MSK_NX51_canctrl_mode_reset_mode) == 0)
    return NX51_CANCTRL_RESULT_WRONG_MODE;

  s_ptCanCtrl->ulCanctrl_err_warning_limit = ulLimit;
  
  return NX51_CANCTRL_RESULT_OKAY;
}

/*****************************************************************************/
/*! Get Error Warning Limit
* \description
*   Returns the warning limit for error counters.
* \class 
*   Status Service Class
* \params
* \return
*   Current err_warning_limit value                                          */
/*****************************************************************************/
uint32_t NX51_CanCtrl_GetWarningLimit( void )
{
  return s_ptCanCtrl->ulCanctrl_err_warning_limit;
}

/*****************************************************************************/
/*! Set RX Error Counter
* \description
*   Sets the value of the RX error counter.
* \class 
*   Control Service Class
* \params
*   ulCnt        [in]   Error counter value
* \return
*   NX51_CANCTRL_RESULT_OKAY      
*   NX51_CANCTRL_RESULT_WRONG_MODE                                           */
/*****************************************************************************/
NX51_CANCTRL_RESULT_E NX51_CanCtrl_SetRxErrorCnt( uint32_t ulCnt )
{
  if ((s_ptCanCtrl->ulCanctrl_mode & MSK_NX51_canctrl_mode_reset_mode) == 0)
    return NX51_CANCTRL_RESULT_WRONG_MODE;

  s_ptCanCtrl->ulCanctrl_rx_error_cnt = ulCnt;
  
  return NX51_CANCTRL_RESULT_OKAY;
}

/*****************************************************************************/
/*! Get RX Error Counter
* \description
*   Returns the value of the RX error counter.
* \class 
*   Status Service Class
* \params
* \return
*   Current rx_err_cnt value                                                 */
/*****************************************************************************/
uint32_t NX51_CanCtrl_GetRxErrorCnt( void )
{
  return s_ptCanCtrl->ulCanctrl_rx_error_cnt;
}

/*****************************************************************************/
/*! Set TX Error Counter
* \description
*   Sets the value of the TX error counter.
* \class 
*   Control Service Class
* \params
*   ulCnt        [in]   Error counter value
* \return
*   NX51_CANCTRL_RESULT_OKAY      
*   NX51_CANCTRL_RESULT_WRONG_MODE                                           */
/*****************************************************************************/
NX51_CANCTRL_RESULT_E NX51_CanCtrl_SetTxErrorCnt( uint32_t ulCnt )
{
  if ((s_ptCanCtrl->ulCanctrl_mode & MSK_NX51_canctrl_mode_reset_mode) == 0)
    return NX51_CANCTRL_RESULT_WRONG_MODE;

  s_ptCanCtrl->ulCanctrl_tx_error_cnt = ulCnt;
  
  return NX51_CANCTRL_RESULT_OKAY;
}

/*****************************************************************************/
/*! Get TX Error Counter
* \description
*   Gets the value of the TX error counter.
* \class 
*   Status Service Class
* \params
* \return
*   Current tx_err_cnt value                                                 */
/*****************************************************************************/
uint32_t NX51_CanCtrl_GetTxErrorCnt( void )
{
  return s_ptCanCtrl->ulCanctrl_tx_error_cnt;
}
